# cli
# Copyright 2010-2011 Karl A. Knechtel.
#
# A simple, reusable command-line interface.
#
# Licensed under the Generic Non-Commercial Copyleft Software License,
# Version 1.1 (hereafter "Licence"). You may not use this file except
# in the ways outlined in the Licence, which you should have received
# along with this file.
#
# Unless required by applicable law or agreed to in writing, software 
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.


from exception import PEBKAC, Done
from command import command, execute, COMMANDS
from text import wrap, display
import re, ast, sys

IE = """*** INTERNAL ERROR ***

Oops, looks like you found a bug. An error log has been saved in the same
folder as the script. Please include that text in your bug report to help fix
the bug. (The error log contains the path to the script on your system; if you
are concerned about privacy, you can censor that if you like.)

Press return to quit."""


IE_NOFILE = """*** INTERNAL ERROR ***

Oops, looks like you found a bug. Unfortunately, an error log could not be
saved. If the bug is reproducible, please explain how to reproduce it as
clearly as possible in your bug report.

Press return to quit."""


NO_WARRANTY = """This program comes with ABSOLUTELY NO WARRANTY.
This is non-commercial, copylefted software; you are permitted to redistribute
it under certain conditions intended to ensure it stays non-commercial.
For details, see the file 'LICENSE.txt' provided with the program."""


HELP_SUGGESTION = "Use the command 'help' for a basic overview."


BASIC_HELP = """Available commands: %s

Commands are not case sensitive. For more information on how to
use a command, try 'help <command>'.

Arguments for commands are normally separated by spaces. To
include a space in an argument, surround the argument in either
single or double quotes (this allows you to embed the other
quote type in the string). Within a quoted string, you may use
\\<character> or \\x<two hex digits> style escapes."""


def crash(log):
	import traceback, time
	try:
		with file('error-%d.txt' % time.time(), 'w') as f:
			traceback.print_exc(file = f)
			f.write('\n\nLOG:\n')
			f.write(log)
	except:
		# Panic! couldn't even write the file
		raw_input(wrap(IE_NOFILE))
	else:
		raw_input(wrap(IE))


pattern = re.compile(u'([^ "\t\n]+|"(?:\\\\.|[^\\\\"\n])*")')
def tokenize(source):
	real = False
	for token in pattern.split(source):
		if real: # i.e., not a separator.
			if token.startswith('"'):
				try: yield ast.literal_eval('u' + token)
				except SyntaxError: raise PEBKAC, "Invalid quoted string."
			else: yield token
		elif '"' in token: raise PEBKAC, "Invalid quoted string."
		real = not real


def parse(user_input):
	return list(tokenize(user_input.strip()))


VERSION = (0, 25)


@command()
def help(state, command_name = None): # This shadows a builtin... not that we need it
	"""Shows information about how to use a command.
	But if you're reading this, you already knew that, right?"""

	if command_name == None:
		display(BASIC_HELP % ', '.join(COMMANDS))
		return

	if command_name in COMMANDS:
		display("Help for '%s' command:\n\n%s" % (
			command_name,
			COMMANDS[command_name].doc()
		))
	else:
		raise PEBKAC, "No such command '%s'. Try 'help' for a list of commands." % command_name


@command()
def quit(state):
	"""Exits the program."""
	raise Done


@command('f!force')
def run(state, script_file, **kwargs):
	"""Treats the contents of the named script_file as a sequence of
	commands, and attempts each one in turn.

	If the '-f' flag is provided, continues to run commands even if
	there is an error with a previous command."""

	try:
		sys.path.append('..')
		from unicode_file import unicode_file
	except:
		display("Warning: Unicode support not available.")
		unicode_file = file

	try:
		with unicode_file(script_file) as f: lines = f.readlines()
	except IOError:
		raise PEBKAC, "File not found."

	for line in lines:
		if line.startswith('#'): continue
		display('> ' + line)
		try:
			execute(state, parse(line))
		except PEBKAC as e:
			if 'f' in kwargs: display("ERROR: " + str(e))
			else: raise


def main(program_name, version, state, preamble):
	display("%s v%s\n\n%s" % (
		program_name,
		'.'.join(str(v) for v in version),
		preamble
	))

	log = ''

	while True:
		try:
			user_input = raw_input('> ').decode(sys.stdin.encoding)
			log += user_input + '\n'
			execute(state, parse(user_input))
		except PEBKAC as e: display("ERROR: " + str(e))
		except (EOFError, KeyboardInterrupt, Done): return
		except: crash(log); return


if __name__ == '__main__':
	main('CLI demo', VERSION, None, NO_WARRANTY + '\n\n' + HELP_SUGGESTION)
